#!/usr/bin/env python
# -*- coding:UTF-8 -*-
'''
    Copyright 2014 Frederic Rible (frible@teaser.fr)

    This wrapper is designed to work with gmoccapy by Norbert Schechner.
    
    It provides automatic tool table update when reading G-Code file
    containing special tags with the following format
    
    ( LINUXCNC/TOOLTABLE, 5, "3.17mm 2 flute ball", 3.17, 1.585, 30.0 )
    
    For CamBam users, add the following line in the post processor "Tool Table Item"
    
    {$comment} LINUXCNC/TOOLTABLE, {$tool.index}, "{$tool.name}", {$tool.diameter}, {$tool.radius}, {$tool.length} {$endcomment}
     
    It implements also a jog-while-paused for M60 by hijackng the pause state
    
    To use it, you should do:
    - make a symlink from gmoccapy.glade to gmoccapy2.glade
      ln -s /usr/share/gscreen/skins/gmoccapy/gmoccapy.glade gmoccapy2.glade
    - install the gmoccapy2_handler.py file in your config directory
    - in your .ini file, replace "DISPLAY = gscreen -c gmoccapy -d"
                            by "DISPLAY = gscreen -c gmoccapy2 -d"
    
    -----
     
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

'''
import hal
import hal_glib
import gobject, gtk
import os
import sys
import pango
import gladevcp.makepins
import gladevcp.tooledit_widget
sys.path.append("./python")
from sharedvar import *
import gtk
import linuxcnc
import gst

sys.path.append("/usr/share/gscreen/skins/gmoccapy")
from gmoccapy_handler import HandlerClass

_MANUAL = 1             # Check for the mode Manual
_AUTO = 2               # Check for the mode Auto
_MDI = 3                # Check for the mode MDI
_RUN = 1                # needed to check if the interpreter is running
_IDLE = 0               # needed to check if the interpreter is idle

def get_handlers(halcomp,builder,useropts,gscreen):
     return [HandlerClass2(halcomp,builder,useropts,gscreen)]

class ToolUpdater:
    def __init__(self, tooleditor):
        self.tooleditor = tooleditor
    
    def UpdateTool(self, toolnumber, diameter , remark):
        self.tooleditor.set_selected_tool(toolnumber)
        self.tooleditor.delete(self)
        print ("add tool %d" % (toolnumber))
        self.tooleditor.add(self, [0,toolnumber,toolnumber,'0','0','0','0','0','0','0','0','0', diameter,'0','0','0',remark])
        
    def reorder(self):        
        # Reorder the tool table     
        liststore = self.tooleditor.model
        lst = []
        r = 0
        for row in liststore:
            values = [ value for value in row ]
            lst.append((r, values[1]))
            r = r+1
        lst = sorted(lst, key=lambda tool: tool[1])
        nr = 0
        new_order = [0] * len(lst)
        for (r,t) in lst:
            new_order[nr] = r
            nr = nr+1
        
        liststore.reorder(new_order)
       
#        for row in liststore:
#            values = [ value for value in row ] 
#            print values      
                                     
    def Parse_GCODE(self, gcode_filename):
        # ( LINUXCNC/TOOLTABLE, 5, "3.17mm 2 flute ball", 3.17, 1.585, 30.0 )
        import re
        modified = False
        pattern = re.compile("\(.*LINUXCNC/TOOLTABLE, *([0-9]+) *, *\"([^\"]*)\" *, *([0-9\.]+) *, *([0-9\.]+) *, *([0-9\.]+) *\)")
        gcode = open(gcode_filename, "r").readlines()
        for rawline in gcode:
            m = re.match(pattern, rawline)
            if (m != None):
                print m.groups()
                self.UpdateTool(int(m.group(1)), m.group(3), m.group(2))
                modified= True
        if modified:
            self.reorder()
        return modified

def new_sourceview_load_file(self, filename=None):
        print "new_sourceview_load_file %s" % (filename)
        if self.h.interpreter == _RUN:
            print "skip"
            return
        old_sourceview_load_file(filename)
        old_gremlin_load(filename)

def new_gremlin_load(self,filename = None):
    pass
#    print "new_gremlin_load %s" % (filename)
#    if self.h.interpreter == _RUN:
#        print "skip"
#        return
#    old_gremlin_load(filename)
        
class HandlerClass2(HandlerClass):
    def __init__(self, halcomp,builder,useropts,gscreen):
        HandlerClass.__init__(self, halcomp,builder,useropts,gscreen)
        self.sharedvar = sharedvar()
        self.sharedvar["RESUME-at-line"] = None
        self._current_file = None
        self.curLine = None                
        self.resume_at_line = None
        self.pause_at_line = None
        self.emcstat = linuxcnc.stat()
        # To avoid error on MDI commands requested after startup by MPG buttons
        # Not sure it realy works !
        self.command.mode( self.linuxcnc.MODE_MANUAL )
        self.command.wait_complete()        
         
    def initialize_widgets(self):
        global old_gremlin_load
        global old_sourceview_load_file
        
        HandlerClass.initialize_widgets(self)    
        
        # New handlers to avoid loading bad G-Code when executing subroutines
        
        instancemethod = type(self.gscreen.widgets.gcode_view.load_file)
        old_sourceview_load_file = self.gscreen.widgets.gcode_view.load_file
        self.gscreen.widgets.gcode_view.h = self
        self.gscreen.widgets.gcode_view.load_file = instancemethod (new_sourceview_load_file, self.gscreen.widgets.gcode_view, type(self.gscreen.widgets.gcode_view))
        
        instancemethod = type(self.gscreen.widgets.gremlin.load)
        old_gremlin_load = self.gscreen.widgets.gremlin.load
        self.gscreen.widgets.gremlin.h = self
        self.gscreen.widgets.gremlin.load = instancemethod (new_gremlin_load, self.gscreen.widgets.gremlin, type(self.gscreen.widgets.gremlin))
        
        self.gscreen.widgets["hal_status"].connect("interp-paused", self.on_hal_status_interp_paused)
        
        self.new_pause_action = gtk.Action('new_pause_action', None, 'Pause', None)
        self.widgets.tbtn_pause.set_related_action(self.new_pause_action)
        self.widgets.tbtn_pause.connect('toggled', self.on_pause_toggled)
        
        self.new_stop_action = gtk.Action('new_stop_action', None, 'Stop', None)
        self.widgets.btn_stop.set_related_action(self.new_stop_action)
        gobject.timeout_add(250, self.my_timer)
        
        self.img_pause_orange = gtk.Image()
        self.img_pause_orange.set_from_file("pause_orange.png")
        self.img_pause_blue = gtk.Image()
        self.img_pause_blue.set_from_file("icon/pause.png")
        self.pause_sound =  "/usr/share/sounds/gnome/default/alerts/glass.ogg"
        
    def initialize_pins(self):
      HandlerClass.initialize_pins(self)
      self.data['mpg-start'] = hal_glib.GPin(self.halcomp.newpin('mpg-start', hal.HAL_BIT, hal.HAL_IN))
      self.data['mpg-start'].connect('value-changed', self.on_mpg_start_changed)
      self.data['mpg-stop'] = hal_glib.GPin(self.halcomp.newpin('mpg-stop', hal.HAL_BIT, hal.HAL_IN))
      self.data['mpg-stop'].connect('value-changed', self.on_mpg_stop_changed)
      self.pin_to_button('mpg-zero-x', self.widgets.btn_zero_x)
      self.pin_to_button('mpg-zero-y', self.widgets.btn_zero_y)
      self.pin_to_button('mpg-zero-z', self.widgets.btn_zero_z)
      self.data['gremlin-refresh'] = hal_glib.GPin(self.halcomp.newpin('gremlin-refresh', hal.HAL_BIT, hal.HAL_IN))
      self.data['gremlin-refresh'].connect('value-changed', self.on_pin_gremlin_refresh_changed)
           
    def pin_to_button(self, pin_name, widget):
        self.data[pin_name] = hal_glib.GPin(self.halcomp.newpin(pin_name, hal.HAL_BIT, hal.HAL_IN))
        self.data[pin_name].connect('value-changed', self.on_pin_to_button_changed)
        self.data[pin_name].widget = widget
                                
    def on_hal_status_interp_paused(self, widget):
        self.resume_at_line = self.sharedvar["RESUME-at-line"]
        self.pause_at_line = self.sharedvar["PAUSE-at-line"]
        print "on_hal_status_interp_paused", self.resume_at_line, self.pause_at_line
        if (self.pause_at_line != None):
            self.gscreen.widgets.gcode_view.set_line_number(self.pause_at_line)
        if (self.resume_at_line != None):   # Got to jog-whule-paused state
            self.widgets.hal_action_stop.emit('activate')
    
    def on_hal_status_interp_run(self,widget):
        HandlerClass.on_hal_status_interp_run(self, widget)
        self.widgets.tbtn_pause.set_sensitive(True)
        
    def on_hal_status_interp_idle(self,widget):
        HandlerClass.on_hal_status_interp_idle(self,widget)
        if (self.resume_at_line == None):
            self.widgets.tbtn_pause.set_active(False)
            self.widgets.tbtn_pause.set_sensitive(False)
        else:
            self.gscreen.add_alarm_entry("pause as idle")
            self.gscreen.widgets.gcode_view.set_line_number(self.pause_at_line)
            self.widgets.tbtn_pause.set_sensitive(True)
            self.widgets.tbtn_pause.set_active(True)
            self.widgets.btn_run.set_sensitive(False)

    def on_pause_toggled(self, widget):
        print "on_pause_toggled", self.widgets.tbtn_pause.get_active(), self.resume_at_line
        if (self.widgets.tbtn_pause.get_active()):  # Go to pause state
            self.widgets.hal_toggleaction_pause.emit('activate')
        else:
            if (self.resume_at_line != None):
                self.widgets.hal_toggleaction_run.set_restart_line(self.resume_at_line)
                self.resume_at_line = None
                self.widgets.hal_toggleaction_run.emit('activate')
                #self.emc.emccommand.auto(self.emc.emc.AUTO_RUN, self.curLine + 1)
            else:
                self.widgets.hal_toggleaction_pause.emit('activate')
    
    def on_btn_stop_clicked(self, widget):
        print "on_btn_stop_clicked"
        if (self.resume_at_line):
            self.resume_at_line = self.sharedvar["RESUME-at-line"] = None
            HandlerClass.on_hal_status_interp_idle(self,widget)
            self.widgets.tbtn_pause.set_sensitive(False)
            self.widgets.tbtn_pause.set_active(False)
        else:
            self.widgets.hal_action_stop.emit('activate')
        self.gscreen.widgets.gcode_view.set_line_number(0)
        
    def on_hal_status_mode_manual( self, widget ):
        print"Manual"
        self.widgets.rbt_manual.set_active( True )
        # setup page will be activated, if we don't leave, the pages will be reset with this call
        if self.widgets.tbtn_setup.get_active() == True:
            return
        if (not self.widgets.tbtn_user_tabs.get_active()):
            self.widgets.ntb_main.set_current_page( 0 )
        else:
            self.widgets.ntb_main.set_current_page( 2 )
            self.widgets.tbtn_fullsize_preview.set_sensitive( False )           
        self.widgets.ntb_button.set_current_page( 0 )
        self.widgets.ntb_info.set_current_page( 0 )
        self.widgets.ntb_jog.set_current_page( 0 )
            
    def on_hal_status_file_loaded(self, widget, filename):
            if self.interpreter == _RUN:
                return
            if (filename != "" and filename != None and filename != self._current_file):
                tu = ToolUpdater(self.widgets.tooledit1)
                if (tu.Parse_GCODE(filename)):
                    self.widgets.tooledit1.save(None)
                    self.widgets.tooledit1.reload(None)
                    message = "Tool table updated by G-CODE"
                    self.gscreen.add_alarm_entry(message)
            HandlerClass.on_hal_status_file_loaded(self, widget, filename)
            self._current_file = filename
    
    def on_btn_zero_z_clicked(self, widget, data=None):
        print "btn_zero_Z_clicked"
        if self.log: self.gscreen.add_alarm_entry("btn_zero_Z_clicked")
        self.gscreen.emc.set_mdi_mode()
		#self.gscreen.mdi_control.set_axis("Z",0)
		#self.gscreen.mdi_control.user_command("#1000=#1001")
        self.gscreen.mdi_control.user_command("o<zero-z>call [0]")
        self.widgets.hal_action_reload.emit("activate")
        self.gscreen.emc.set_manual_mode()
        self.gscreen.add_alarm_entry("set ref tool length")

    def on_btn_set_value_clicked( self, widget, data = None ):
        if widget == self.widgets.btn_set_value_x:
            axis = "x"
        elif widget == self.widgets.btn_set_value_y:
            axis = "y"
        elif widget == self.widgets.btn_set_value_z:
            axis = "z"
        else:
            axis = "Unknown"
            self.gscreen.add_alarm_entry( _( "Offset %s could not be set, because off unknown axis" ) % axis )
            return
        self.gscreen.add_alarm_entry( "btn_set_value_%s_clicked" % axis )
        preset = self.prefs.getpref( "offset_axis_%s" % axis, 0, float )
        offset = self.entry_dialog( data = preset, header = _( "Enter value for axis %s" ) % axis,
                                   label = _( "Set axis %s to:" ) % axis, integer = False )
        if offset == "CANCEL" or offset == "ERROR":
            return
        if offset != False or offset == 0:
            self.gscreen.add_alarm_entry( _( "offset {0} set to {1:f}" ).format( axis, offset ) )
            self.command.mode( self.linuxcnc.MODE_MDI )
            self.command.wait_complete()
            if (axis != "z"):
                self.gscreen.mdi_control.set_axis( axis, offset )
            else:
                self.gscreen.mdi_control.user_command("o<zero-z>call [%f]" % (offset))
            self.widgets.hal_action_reload.emit( "activate" )
            self.command.mode( self.linuxcnc.MODE_MANUAL )
            self.command.wait_complete()
            self.prefs.putpref( "offset_axis_%s" % axis, offset, float )
        else:
            print( _( "Conversion error in btn_set_value" ) )
            self.gscreen.add_alarm_entry( _( "Offset conversion error because off wrong entry" ) )
            self.gscreen.warning_dialog( _( "Conversion error in btn_set_value!" ), True,
                                        _( "Please enter only numerical values\nValues have not been applied" ) )
            
    def on_mpg_stop_changed(self,hal_object):
        h = self.gscreen.halcomp
        st = h['mpg-stop']
        print "mpg stop change", st
        if not st: return
        self.widgets.btn_stop.emit("clicked")

    def on_mpg_start_changed(self,hal_object):
        h = self.gscreen.halcomp
        st = h['mpg-start']
        print "mpg start change", st
        if not st: return
        if self.resume_at_line != None: # We are in jog-while-paused state
            self.widgets.tbtn_pause.emit("clicked")
        else:
            if self.stat.task_mode == _AUTO:
                if self.interpreter == _IDLE:
                    self.widgets.btn_run.emit("clicked")
                else:
       	            self.widgets.tbtn_pause.emit("clicked")
    
    def on_pin_to_button_changed(self, hal_object):
        print "on_pin_to_button_changed", hal_object, hal_object.widget, hal_object.value
        if not hal_object.value:
            return
        hal_object.widget.emit("clicked")
    
    def on_pin_gremlin_refresh_changed(self, hal_object):
        print "on_pin_gremlin_refreshed_changed", hal_object, hal_object.value
        self.widgets.hal_action_reload.emit( "activate" )

    def my_timer(self):
        self.emcstat.poll()
        img = self.widgets.tbtn_pause.get_image()
        if img == self.img_pause_orange:
            self.widgets.tbtn_pause.set_image(self.img_pause_blue)
        elif self.resume_at_line != None or self.emcstat.interp_state == 3:
            self.widgets.tbtn_pause.set_image(self.img_pause_orange)
            self.gscreen.audio.set_sound(self.pause_sound)
            self.gscreen.audio.player.set_state(gst.STATE_PLAYING)
        return True
                	
import unittest
import shutil

class TestUM(unittest.TestCase):

    def setUp(self):
        pass

    def test_insert(self):
        t = gladevcp.ToolEdit()  
        print os.getcwd()
        shutil.copy("tool.tbl", "my.tbl")
        t.set_filename("my.tbl")
        tu = ToolUpdater(t)            
        tu.UpdateTool(17, 5.0, "5mm 1 flute")
        tu.UpdateTool(30, 10.0, "10mm ball")
        tu.UpdateTool(22, 15.0, "15mm ball")
        tu.reorder()
        t.save(self)
        pass

    def test_parse(self):
        t = gladevcp.ToolEdit()
        print os.getcwd()
        shutil.copy("tool.tbl", "my.tbl")
        t.set_filename("my.tbl")
        tu = ToolUpdater(t)
        print tu.Parse_GCODE("GCODE_with_tooltable.ngc")
        print t.get_toolinfo(2)
        pass
           
if __name__ == '__main__':
    unittest.main()

